<?php
/**
 * License Status API Endpoint
 * POST /api/v1/status
 * Detaylı lisans ve aktivasyon bilgisi döndürür
 * 
 * @package SWR_License_Server
 * @since 1.0.0
 */

require_once __DIR__ . '/../../includes/config.php';
require_once __DIR__ . '/../../includes/class-database.php';
require_once __DIR__ . '/../../includes/class-license.php';
require_once __DIR__ . '/../../includes/class-api-handler.php';

$api = new APIHandler();
$data = $api->handleRequest();

// Gerekli alanlar
if (empty($data['license_key'])) {
    $api->sendError('missing_field', 'Lisans anahtarı gerekli', 400);
}

// Lisansı getir
$license = new License();
$lic = $license->getByKey($data['license_key']);

if (!$lic) {
    $api->sendError('invalid_license', 'Geçersiz lisans anahtarı', 404);
}

// Aktivasyonları getir
$db = Database::getInstance();
$activations = $db->select(
    "SELECT domain, site_name, status, created_at, last_check_at FROM activations WHERE license_id = :id",
    ['id' => $lic['id']]
);

$active_count = 0;
$sites = [];

foreach ($activations as $act) {
    if ($act['status'] === 'active') {
        $active_count++;
    }
    $sites[] = [
        'domain' => $act['domain'],
        'site_name' => $act['site_name'],
        'status' => $act['status'],
        'activated_at' => $act['created_at'],
        'last_check' => $act['last_check_at']
    ];
}

// Lisans tipi bilgisi
$type_info = LICENSE_TYPES[$lic['license_type']] ?? LICENSE_TYPES['starter'];

// Süre kontrolü
$is_expired = false;
$days_remaining = null;

if ($lic['expires_at']) {
    $expiry = strtotime($lic['expires_at']);
    $is_expired = $expiry < time();
    
    if (!$is_expired) {
        $days_remaining = ceil(($expiry - time()) / 86400);
    }
}

// Log kaydet
$api->logRequest(
    'status',
    $lic['id'],
    $data['domain'] ?? 'unknown',
    'success',
    ''
);

$api->sendSuccess([
    'license' => [
        'key' => $lic['license_key'],
        'type' => $lic['license_type'],
        'type_name' => $type_info['name'],
        'status' => $lic['status'],
        'is_lifetime' => $type_info['period'] === 'lifetime',
        'expires_at' => $lic['expires_at'],
        'is_expired' => $is_expired,
        'days_remaining' => $days_remaining,
        'created_at' => $lic['created_at'],
        'activated_at' => $lic['activated_at']
    ],
    'sites' => [
        'used' => $active_count,
        'allowed' => $lic['max_sites'],
        'is_exceeded' => $active_count > $lic['max_sites'],
        'list' => $sites
    ],
    'customer' => [
        'name' => $lic['customer_name'],
        'email' => $lic['customer_email']
    ],
    'features' => PRO_FEATURES
]);
