<?php
/**
 * API Handler Class
 * API isteklerini işler, güvenlik kontrolleri yapar
 * 
 * @package SWR_License_Server
 * @since 1.0.0
 */

class APIHandler {
    
    private $db;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * İsteği işle
     */
    public function handleRequest() {
        // CORS headers
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: POST, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, X-SWR-Product, X-SWR-Version');
        header('Content-Type: application/json; charset=utf-8');
        
        // OPTIONS request (preflight)
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            http_response_code(200);
            exit;
        }
        
        // Sadece POST kabul et
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->sendError('method_not_allowed', 'Sadece POST istekleri kabul edilir', 405);
        }
        
        // Rate limiting
        if (!$this->checkRateLimit()) {
            $this->sendError('rate_limit_exceeded', 'Çok fazla istek. Lütfen bekleyin.', 429);
        }
        
        // JSON body al
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->sendError('invalid_json', 'Geçersiz JSON verisi', 400);
        }
        
        return $data;
    }
    
    /**
     * İmza doğrulama
     */
    public function verifySignature($data) {
        if (!isset($data['signature'])) {
            return false;
        }
        
        $signature = $data['signature'];
        unset($data['signature']);
        
        ksort($data);
        $string = http_build_query($data);
        $expected = hash_hmac('sha256', $string, API_SECRET_KEY);
        
        return hash_equals($expected, $signature);
    }
    
    /**
     * Rate limiting kontrolü
     */
    private function checkRateLimit() {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $cache_key = 'rate_limit_' . md5($ip);
        
        // Basit dosya tabanlı rate limiting
        $cache_file = sys_get_temp_dir() . '/' . $cache_key;
        
        $requests = 0;
        $window_start = time();
        
        if (file_exists($cache_file)) {
            $cache_data = json_decode(file_get_contents($cache_file), true);
            if ($cache_data && isset($cache_data['start']) && (time() - $cache_data['start']) < RATE_LIMIT_WINDOW) {
                $requests = $cache_data['count'];
                $window_start = $cache_data['start'];
            }
        }
        
        $requests++;
        
        if ($requests > RATE_LIMIT_REQUESTS) {
            return false;
        }
        
        file_put_contents($cache_file, json_encode([
            'count' => $requests,
            'start' => $window_start
        ]));
        
        return true;
    }
    
    /**
     * API log kaydet
     */
    public function logRequest($endpoint, $license_id, $domain, $status, $message = '') {
        $this->db->insert('api_logs', [
            'license_id' => $license_id,
            'endpoint' => $endpoint,
            'domain' => $domain,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'request_data' => json_encode($_POST),
            'response_status' => $status,
            'response_message' => $message
        ]);
    }
    
    /**
     * Başarılı yanıt gönder
     */
    public function sendSuccess($data = []) {
        $response = array_merge(['success' => true], $data);
        echo json_encode($response);
        exit;
    }
    
    /**
     * Hata yanıtı gönder
     */
    public function sendError($error, $message, $http_code = 400) {
        http_response_code($http_code);
        echo json_encode([
            'success' => false,
            'error' => $error,
            'message' => $message
        ]);
        exit;
    }
    
    /**
     * IP adresi al
     */
    public function getClientIP() {
        $headers = [
            'HTTP_CF_CONNECTING_IP', // Cloudflare
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        ];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ips = explode(',', $_SERVER[$header]);
                $ip = trim($ips[0]);
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return 'unknown';
    }
}
