<?php
/**
 * License Class
 * Lisans oluşturma, doğrulama ve yönetim işlemleri
 * 
 * @package SWR_License_Server
 * @since 1.0.0
 */

class License {
    
    private $db;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->db = Database::getInstance();
    }
    
    /**
     * Yeni lisans oluştur
     */
    public function create($data) {
        $license_key = $this->generateLicenseKey();
        $license_type = $data['license_type'] ?? 'starter';
        $type_info = LICENSE_TYPES[$license_type] ?? LICENSE_TYPES['starter'];
        
        // Bitiş tarihi hesapla
        $expires_at = null;
        if ($type_info['period'] === 'yearly') {
            $expires_at = date('Y-m-d H:i:s', strtotime('+1 year'));
        }
        // lifetime için null kalır
        
        $license_data = [
            'license_key' => $license_key,
            'license_type' => $license_type,
            'status' => 'active',
            'max_sites' => $type_info['sites'],
            'customer_name' => $data['customer_name'] ?? null,
            'customer_email' => $data['customer_email'],
            'customer_phone' => $data['customer_phone'] ?? null,
            'payment_id' => $data['payment_id'] ?? null,
            'payment_method' => $data['payment_method'] ?? 'paynet',
            'amount' => $type_info['price'],
            'currency' => 'USD',
            'expires_at' => $expires_at,
            'notes' => $data['notes'] ?? null
        ];
        
        $id = $this->db->insert('licenses', $license_data);
        
        return [
            'id' => $id,
            'license_key' => $license_key,
            'license_type' => $license_type,
            'expires_at' => $expires_at
        ];
    }
    
    /**
     * Lisans anahtarı oluştur
     * Format: SWR-XXXX-XXXX-XXXX-XXXX
     */
    public function generateLicenseKey() {
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        
        do {
            $key = 'SWR';
            for ($i = 0; $i < 4; $i++) {
                $key .= '-';
                for ($j = 0; $j < 4; $j++) {
                    $key .= $chars[random_int(0, strlen($chars) - 1)];
                }
            }
        } while ($this->keyExists($key));
        
        return $key;
    }
    
    /**
     * Lisans anahtarı var mı kontrol et
     */
    public function keyExists($key) {
        return $this->db->count('licenses', 'license_key = :key', ['key' => $key]) > 0;
    }
    
    /**
     * Lisans getir (key ile)
     */
    public function getByKey($license_key) {
        return $this->db->selectOne(
            "SELECT * FROM licenses WHERE license_key = :key",
            ['key' => $license_key]
        );
    }
    
    /**
     * Lisans getir (ID ile)
     */
    public function getById($id) {
        return $this->db->selectOne(
            "SELECT * FROM licenses WHERE id = :id",
            ['id' => $id]
        );
    }
    
    /**
     * Lisans doğrula
     */
    public function verify($license_key, $domain) {
        $license = $this->getByKey($license_key);
        
        if (!$license) {
            return [
                'valid' => false,
                'error' => 'invalid_license',
                'message' => 'Geçersiz lisans anahtarı'
            ];
        }
        
        // Durum kontrolü
        if ($license['status'] !== 'active') {
            return [
                'valid' => false,
                'error' => 'license_' . $license['status'],
                'message' => 'Lisans ' . $this->getStatusText($license['status'])
            ];
        }
        
        // Süre kontrolü
        if ($license['expires_at'] && strtotime($license['expires_at']) < time()) {
            // Lisansı expired olarak güncelle
            $this->db->update('licenses', ['status' => 'expired'], 'id = :id', ['id' => $license['id']]);
            
            return [
                'valid' => false,
                'error' => 'license_expired',
                'message' => 'Lisans süresi dolmuş'
            ];
        }
        
        // Aktivasyonları getir
        $activations = $this->getActivations($license['id']);
        $sites_used = count($activations);
        
        // Bu domain aktif mi kontrol et
        $domain_active = false;
        foreach ($activations as $activation) {
            if ($activation['domain'] === $domain && $activation['status'] === 'active') {
                $domain_active = true;
                break;
            }
        }
        
        // Son doğrulama tarihini güncelle
        $this->db->update(
            'licenses',
            ['last_verified_at' => date('Y-m-d H:i:s')],
            'id = :id',
            ['id' => $license['id']]
        );
        
        return [
            'valid' => true,
            'license_type' => $license['license_type'],
            'expires_at' => $license['expires_at'],
            'sites_used' => $sites_used,
            'sites_allowed' => $license['max_sites'],
            'domain_active' => $domain_active,
            'features' => PRO_FEATURES
        ];
    }
    
    /**
     * Site aktivasyonu
     */
    public function activate($license_key, $domain, $extra_data = []) {
        $license = $this->getByKey($license_key);
        
        if (!$license) {
            return [
                'success' => false,
                'error' => 'invalid_license',
                'message' => 'Geçersiz lisans anahtarı'
            ];
        }
        
        // Durum kontrolü
        if ($license['status'] !== 'active') {
            return [
                'success' => false,
                'error' => 'license_' . $license['status'],
                'message' => 'Lisans ' . $this->getStatusText($license['status'])
            ];
        }
        
        // Süre kontrolü
        if ($license['expires_at'] && strtotime($license['expires_at']) < time()) {
            return [
                'success' => false,
                'error' => 'license_expired',
                'message' => 'Lisans süresi dolmuş'
            ];
        }
        
        // Bu domain zaten aktif mi?
        $existing = $this->db->selectOne(
            "SELECT * FROM activations WHERE license_id = :license_id AND domain = :domain AND status = 'active'",
            ['license_id' => $license['id'], 'domain' => $domain]
        );
        
        if ($existing) {
            // Mevcut aktivasyonu güncelle
            $this->db->update(
                'activations',
                [
                    'last_check_at' => date('Y-m-d H:i:s'),
                    'plugin_version' => $extra_data['version'] ?? null,
                    'wp_version' => $extra_data['wp_version'] ?? null,
                    'php_version' => $extra_data['php_version'] ?? null
                ],
                'id = :id',
                ['id' => $existing['id']]
            );
            
            $activations = $this->getActivations($license['id']);
            
            return [
                'success' => true,
                'message' => 'Aktivasyon güncellendi',
                'license_type' => $license['license_type'],
                'expires_at' => $license['expires_at'],
                'sites_used' => count($activations),
                'sites_allowed' => $license['max_sites'],
                'features' => PRO_FEATURES
            ];
        }
        
        // Site limiti kontrolü
        $activations = $this->getActivations($license['id']);
        if (count($activations) >= $license['max_sites']) {
            return [
                'success' => false,
                'error' => 'site_limit_reached',
                'message' => 'Site limiti dolmuş (' . $license['max_sites'] . ' site)',
                'sites_used' => count($activations),
                'sites_allowed' => $license['max_sites']
            ];
        }
        
        // Yeni aktivasyon
        $this->db->insert('activations', [
            'license_id' => $license['id'],
            'domain' => $domain,
            'domain_hash' => $extra_data['domain_hash'] ?? hash('sha256', $domain),
            'site_name' => $extra_data['site_name'] ?? null,
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
            'status' => 'active',
            'plugin_version' => $extra_data['version'] ?? null,
            'wp_version' => $extra_data['wp_version'] ?? null,
            'php_version' => $extra_data['php_version'] ?? null
        ]);
        
        // Lisansın ilk aktivasyonu ise activated_at güncelle
        if (!$license['activated_at']) {
            $this->db->update('licenses', ['activated_at' => date('Y-m-d H:i:s')], 'id = :id', ['id' => $license['id']]);
        }
        
        $activations = $this->getActivations($license['id']);
        
        return [
            'success' => true,
            'message' => 'Lisans başarıyla aktive edildi',
            'license_type' => $license['license_type'],
            'expires_at' => $license['expires_at'],
            'sites_used' => count($activations),
            'sites_allowed' => $license['max_sites'],
            'features' => PRO_FEATURES
        ];
    }
    
    /**
     * Site deaktivasyonu
     */
    public function deactivate($license_key, $domain) {
        $license = $this->getByKey($license_key);
        
        if (!$license) {
            return [
                'success' => false,
                'error' => 'invalid_license',
                'message' => 'Geçersiz lisans anahtarı'
            ];
        }
        
        $affected = $this->db->update(
            'activations',
            [
                'status' => 'deactivated',
                'deactivated_at' => date('Y-m-d H:i:s')
            ],
            "license_id = :license_id AND domain = :domain AND status = 'active'",
            ['license_id' => $license['id'], 'domain' => $domain]
        );
        
        if ($affected > 0) {
            return [
                'success' => true,
                'message' => 'Lisans deaktive edildi'
            ];
        }
        
        return [
            'success' => false,
            'error' => 'activation_not_found',
            'message' => 'Aktif aktivasyon bulunamadı'
        ];
    }
    
    /**
     * Lisans aktivasyonlarını getir
     */
    public function getActivations($license_id) {
        return $this->db->select(
            "SELECT * FROM activations WHERE license_id = :license_id AND status = 'active'",
            ['license_id' => $license_id]
        );
    }
    
    /**
     * Lisans yenileme
     */
    public function renew($license_key, $payment_id = null) {
        $license = $this->getByKey($license_key);
        
        if (!$license) {
            return ['success' => false, 'message' => 'Lisans bulunamadı'];
        }
        
        $type_info = LICENSE_TYPES[$license['license_type']] ?? LICENSE_TYPES['starter'];
        
        // Lifetime ise yenileme gerekmez
        if ($type_info['period'] === 'lifetime') {
            return ['success' => false, 'message' => 'Ömür boyu lisans yenileme gerektirmez'];
        }
        
        // Yeni bitiş tarihi hesapla
        $current_expiry = $license['expires_at'] ? strtotime($license['expires_at']) : time();
        $base_date = max($current_expiry, time());
        $new_expiry = date('Y-m-d H:i:s', strtotime('+1 year', $base_date));
        
        $this->db->update(
            'licenses',
            [
                'expires_at' => $new_expiry,
                'status' => 'active',
                'payment_id' => $payment_id
            ],
            'id = :id',
            ['id' => $license['id']]
        );
        
        return [
            'success' => true,
            'message' => 'Lisans yenilendi',
            'expires_at' => $new_expiry
        ];
    }
    
    /**
     * Durum metni
     */
    private function getStatusText($status) {
        $texts = [
            'active' => 'aktif',
            'expired' => 'süresi dolmuş',
            'suspended' => 'askıya alınmış',
            'revoked' => 'iptal edilmiş'
        ];
        return $texts[$status] ?? $status;
    }
    
    /**
     * Tüm lisansları getir (admin panel için)
     */
    public function getAll($page = 1, $per_page = 20, $filters = []) {
        $offset = ($page - 1) * $per_page;
        $where = '1=1';
        $params = [];
        
        if (!empty($filters['status'])) {
            $where .= ' AND status = :status';
            $params['status'] = $filters['status'];
        }
        
        if (!empty($filters['search'])) {
            $where .= ' AND (license_key LIKE :search OR customer_email LIKE :search OR customer_name LIKE :search)';
            $params['search'] = '%' . $filters['search'] . '%';
        }
        
        $total = $this->db->count('licenses', $where, $params);
        
        $licenses = $this->db->select(
            "SELECT * FROM licenses WHERE {$where} ORDER BY created_at DESC LIMIT {$per_page} OFFSET {$offset}",
            $params
        );
        
        return [
            'data' => $licenses,
            'total' => $total,
            'pages' => ceil($total / $per_page),
            'current_page' => $page
        ];
    }
}
