<?php
/**
 * Paynet Payment Integration
 * Paynet ödeme sistemi entegrasyonu
 * 
 * @package SWR_License_Server
 * @since 1.0.0
 */

class Paynet {
    
    private $secret_key;
    private $public_key;
    private $test_mode;
    private $base_url;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->secret_key = PAYNET_SECRET_KEY;
        $this->public_key = PAYNET_PUBLIC_KEY;
        $this->test_mode = PAYNET_TEST_MODE;
        
        // API URL
        $this->base_url = $this->test_mode 
            ? 'https://pts-api.paynet.com.tr/v1' 
            : 'https://api.paynet.com.tr/v1';
    }
    
    /**
     * Ödeme başlat
     */
    public function createPayment($data) {
        $payment_data = [
            'amount' => $data['amount'] * 100, // Kuruş cinsinden
            'currency' => 'USD',
            'order_id' => $data['order_id'],
            'description' => $data['description'] ?? 'Smart Word Replacer License',
            'return_url' => $data['return_url'],
            'cancel_url' => $data['cancel_url'],
            'customer' => [
                'email' => $data['email'],
                'name' => $data['name'] ?? ''
            ]
        ];
        
        // Paynet API isteği
        $response = $this->request('POST', '/payments', $payment_data);
        
        return $response;
    }
    
    /**
     * Ödeme durumu sorgula
     */
    public function getPaymentStatus($payment_id) {
        return $this->request('GET', '/payments/' . $payment_id);
    }
    
    /**
     * Ödeme doğrula (callback)
     */
    public function verifyPayment($transaction_id, $hash) {
        // Hash doğrulama
        $expected_hash = hash_hmac('sha256', $transaction_id, $this->secret_key);
        
        if (!hash_equals($expected_hash, $hash)) {
            return ['success' => false, 'error' => 'Invalid hash'];
        }
        
        // Ödeme durumunu kontrol et
        $payment = $this->getPaymentStatus($transaction_id);
        
        if ($payment && $payment['status'] === 'completed') {
            return [
                'success' => true,
                'transaction_id' => $transaction_id,
                'amount' => $payment['amount'] / 100,
                'currency' => $payment['currency']
            ];
        }
        
        return ['success' => false, 'error' => 'Payment not completed'];
    }
    
    /**
     * API isteği gönder
     */
    private function request($method, $endpoint, $data = null) {
        $url = $this->base_url . $endpoint;
        
        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $this->secret_key,
            'X-API-Key: ' . $this->public_key
        ];
        
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_SSL_VERIFYPEER => true
        ]);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['success' => false, 'error' => $error];
        }
        
        $result = json_decode($response, true);
        
        if ($http_code >= 200 && $http_code < 300) {
            return $result;
        }
        
        return [
            'success' => false,
            'error' => $result['message'] ?? 'Unknown error',
            'http_code' => $http_code
        ];
    }
    
    /**
     * Sipariş ID oluştur
     */
    public function generateOrderId() {
        return 'SWR-' . date('Ymd') . '-' . strtoupper(substr(uniqid(), -8));
    }
}
